/**
 * Serviço de Notificações via Telegram
 * Envia mensagens sobre o status da automação usando a API oficial do Telegram Bot.
 * AGORA COM CONTROLE REMOTO (Polling + Proxy)
 */

const https = require('https');
const http = require('http'); // Para chamadas locais (Proxy)

let config = {
    botToken: '',
    chatId: '',
    currentPort: 3008, // Default, será atualizado no init
    instanceUrl: ''
};

let pollInterval = null;
let lastUpdateId = 0;

function init(notificationConfig) {
    if (notificationConfig && notificationConfig.botToken && notificationConfig.chatId) {
        config = {
            botToken: notificationConfig.botToken,
            chatId: notificationConfig.chatId,
            currentPort: notificationConfig.currentPort || 3008,
            instanceUrl: notificationConfig.instanceUrl
        };
        console.log(`[Notification] Serviço Telegram configurado na porta ${config.currentPort}.`);

        // ESTRATÉGIA DE POLLING: Apenas o painel 3008 (Master) faz o polling.
        // Isso evita conflitos de offset na API do Telegram (Race Conditions).
        // O Painel 3008 recebe os comandos e distribui via Proxy HTTP para os outros (3009, 3010, etc).
        if (parseInt(config.currentPort) === 3008) {
            console.log('👑 [Notification] Este é o Painel MASTER (3008). Iniciando Polling...');
            startPolling();
        } else {
            console.log('👂 [Notification] Painel SLAVE. Aguardando comandos via Proxy HTTP (sem polling).');
        }
    }
}

// ==================== POLLING LOGIC ====================

function startPolling() {
    if (pollInterval) clearInterval(pollInterval);

    // Polling a cada 2 segundos
    pollInterval = setInterval(async () => {
        try {
            const updates = await getUpdates(lastUpdateId + 1);
            if (updates && updates.length > 0) {
                for (const update of updates) {
                    lastUpdateId = update.update_id;
                    await processUpdate(update);
                }
            }
        } catch (e) {
            console.error(`[Polling Error] ${e.message}`);
        }
    }, 2000);
}

async function getUpdates(offset) {
    if (!config.botToken) return [];

    const url = `https://api.telegram.org/bot${config.botToken}/getUpdates?offset=${offset}&timeout=1`;

    return new Promise((resolve, reject) => {
        https.get(url, (res) => {
            let data = '';
            res.on('data', chunk => data += chunk);
            res.on('end', () => {
                try {
                    const parsed = JSON.parse(data);
                    if (parsed.ok) resolve(parsed.result);
                    else resolve([]);
                } catch (e) { resolve([]); }
            });
        }).on('error', (e) => resolve([]));
    });
}

async function processUpdate(update) {
    // 1. Comandos de Texto (/painel)
    if (update.message && update.message.text) {
        const text = update.message.text;
        if (text === '/painel' || text === '/start' || text === '/menu') {
            console.log('👑 [Master] Recebido comando /painel');
            await sendControlPanel(update.message.chat.id);
        }
    }

    // 2. Botões (Callback Query)
    if (update.callback_query) {
        const data = update.callback_query.data;
        const chatId = update.callback_query.message.chat.id;

        console.log(`[Action] Recebido clique: ${data}`);

        // Responder ao Telegram para parar o loading do botão
        await answerCallbackQuery(update.callback_query.id);

        // Parsear ação: acao_porta (ex: start_3008, stop_3010)
        const [action, targetPortStr] = data.split('_');
        const targetPort = parseInt(targetPortStr);

        if (targetPort && (action === 'start' || action === 'stop')) {
            await executeProxyCommand(action, targetPort, chatId);
        }
    }
}

// ==================== PROXY LOGIC ====================

async function executeProxyCommand(action, targetPort, chatId) {
    const isLocal = targetPort === parseInt(config.currentPort);

    console.log(`[Proxy] Comando ${action} para ${targetPort}. Eu sou ${config.currentPort}. É local? ${isLocal}`);

    if (isLocal) {
        // Executar localmente
        await callLocalApi(action, targetPort);
        await sendTelegram(`✅ Comando <b>${action.toUpperCase()}</b> executado no Painel <b>${targetPort}</b> (Local).`);
    } else {
        // Enviar requisição para a outra instância
        try {
            await callRemoteApi(action, targetPort);
            await sendTelegram(`📡 Comando <b>${action.toUpperCase()}</b> encaminhado para Painel <b>${targetPort}</b>.`);
        } catch (e) {
            await sendTelegram(`❌ Falha ao comunicar com Painel <b>${targetPort}</b>: ${e.message}`);
        }
    }
}

async function callLocalApi(action, port) {
    return callRemoteApi(action, port); // Mesma lógica http://localhost
}

async function callRemoteApi(action, port) {
    return new Promise((resolve, reject) => {
        const postData = JSON.stringify({
            // AGORA: Enviamos payload vazio para sinalizar "Remote Click".
            // O servidor vai receber isso, ver que está vazio e mandar um broadcast
            // para o navegador clicar no botão "Iniciar".
            // NÃO mandamos mais dummy data.
        });

        const options = {
            hostname: 'localhost',
            port: port,
            path: `/api/remix/${action}`,
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Content-Length': Buffer.byteLength(postData)
            },
            timeout: 5000
        };

        const req = http.request(options, (res) => {
            if (res.statusCode >= 200 && res.statusCode < 300) {
                resolve(true);
            } else {
                reject(new Error(`Status ${res.statusCode}`));
            }
        });

        req.on('error', (e) => reject(e));

        req.write(postData);
        req.end();
    });
}

// ==================== UI LOGIC ====================

async function sendControlPanel(chatId) {
    const keyboard = {
        inline_keyboard: [
            [
                { text: '🟢 Start 3008', callback_data: 'start_3008' },
                { text: '🔴 Stop 3008', callback_data: 'stop_3008' }
            ],
            [
                { text: '🟢 Start 3009', callback_data: 'start_3009' },
                { text: '🔴 Stop 3009', callback_data: 'stop_3009' }
            ],
            [
                { text: '🟢 Start 3010', callback_data: 'start_3010' },
                { text: '🔴 Stop 3010', callback_data: 'stop_3010' }
            ],
            [
                { text: '🟢 Start 3011', callback_data: 'start_3011' },
                { text: '🔴 Stop 3011', callback_data: 'stop_3011' }
            ]
        ]
    };

    await sendTelegram('🎛 <b>PAINEL DE CONTROLE LOVABLE</b>\nSelecione uma ação:', keyboard);
}

async function answerCallbackQuery(callbackQueryId) {
    if (!config.botToken) return;
    const url = `https://api.telegram.org/bot${config.botToken}/answerCallbackQuery?callback_query_id=${callbackQueryId}`;
    https.get(url).on('error', () => { });
}

// ==================== SEND LOGIC (EXISTING) ====================

async function sendTelegram(message, replyMarkup = null) {
    if (!config.botToken || !config.chatId) {
        console.log('[Notification DEBUG] FALHA: Telegram não configurado.');
        return;
    }

    try {
        const encodedMessage = encodeURIComponent(message);
        let url = `https://api.telegram.org/bot${config.botToken}/sendMessage?chat_id=${config.chatId}&text=${encodedMessage}&parse_mode=HTML`;

        if (replyMarkup) {
            url += `&reply_markup=${encodeURIComponent(JSON.stringify(replyMarkup))}`;
        }

        return new Promise((resolve, reject) => {
            https.get(url, (res) => {
                let data = '';
                res.on('data', chunk => data += chunk);
                res.on('end', () => resolve(true));
            }).on('error', (e) => resolve(false));
        });

    } catch (error) {
        console.error(`[Notification] Erro interno Telegram: ${error.message}`);
        return false;
    }
}

module.exports = {
    init,
    sendTelegram
};
